"use client"

import { useState, useEffect } from "react"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Input } from "@/components/ui/input"
import { Badge } from "@/components/ui/badge"
import { Avatar, AvatarFallback } from "@/components/ui/avatar"
import { Alert, AlertDescription } from "@/components/ui/alert"
import { Clock, Users, Trophy, RotateCcw, Eye, Wifi, WifiOff, Share2, Copy, Home, RefreshCw } from "lucide-react"
import { useGame } from "@/hooks/use-game"
import { useRouter } from "next/navigation"

const TIME_OPTIONS = [
  { value: 0, label: "On Time! 🎉", range: "0 min" },
  { value: 2, label: "Fashionably Late", range: "1-3 min" },
  { value: 5, label: "Classic Kostas", range: "4-6 min" },
  { value: 8, label: "Getting Worried", range: "7-9 min" },
  { value: 12, label: "Where is He?", range: "10-14 min" },
  { value: 18, label: "Send Search Party", range: "15-20 min" },
  { value: 25, label: "Kostas.exe Stopped", range: "20+ min" },
]

interface GameRoomProps {
  gameId: string
}

export default function GameRoom({ gameId }: GameRoomProps) {
  const router = useRouter()
  const { gameState, isConnected, error, isInitialized, sendAction, reconnect } = useGame(gameId)
  const [newPlayerName, setNewPlayerName] = useState("")
  const [currentPlayerId, setCurrentPlayerId] = useState<string | null>(null)
  const [showShareSuccess, setShowShareSuccess] = useState(false)

  // Get or create player ID
  useEffect(() => {
    const stored = localStorage.getItem(`kostas-poker-player-${gameId}`)
    if (stored) {
      setCurrentPlayerId(stored)
    }
  }, [gameId])

  const addPlayer = async () => {
    if (!newPlayerName.trim()) return

    const playerId = Date.now().toString() + Math.random().toString(36).substr(2, 9)

    try {
      await sendAction("ADD_PLAYER", {
        id: playerId,
        name: newPlayerName.trim(),
      })

      localStorage.setItem(`kostas-poker-player-${gameId}`, playerId)
      setCurrentPlayerId(playerId)
      setNewPlayerName("")
    } catch (err) {
      console.error("Failed to add player:", err)
    }
  }

  const placeBet = async (bet: number) => {
    if (!currentPlayerId) return

    try {
      await sendAction("PLACE_BET", {
        playerId: currentPlayerId,
        bet,
      })
    } catch (err) {
      console.error("Failed to place bet:", err)
    }
  }

  const shareGame = async () => {
    const gameUrl = `${window.location.origin}/game/${gameId}`

    if (navigator.share) {
      try {
        await navigator.share({
          title: "Join Kostas Late Poker!",
          text: `Join our betting game on how late Kostas will be! Room: ${gameId}`,
          url: gameUrl,
        })
      } catch (err) {
        // Fallback to clipboard
        copyToClipboard(gameUrl)
      }
    } else {
      copyToClipboard(gameUrl)
    }
  }

  const copyToClipboard = async (text: string) => {
    try {
      await navigator.clipboard.writeText(text)
      setShowShareSuccess(true)
      setTimeout(() => setShowShareSuccess(false), 3000)
    } catch (err) {
      console.error("Failed to copy to clipboard:", err)
    }
  }

  const startBetting = () => sendAction("START_BETTING")
  const revealBets = () => sendAction("REVEAL_BETS")
  const finishGame = (actualTime: number) => sendAction("FINISH_GAME", { actualTime })
  const resetGame = () => sendAction("RESET_GAME")
  const goHome = () => router.push("/")

  // Show loading state while initializing
  if (!isInitialized || !gameState) {
    return (
      <div className="min-h-screen bg-gradient-to-br from-purple-900 via-blue-900 to-indigo-900 flex items-center justify-center">
        <Card className="bg-white/10 border-white/20">
          <CardContent className="p-8 text-center">
            <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-white mx-auto mb-4"></div>
            <p className="text-white text-lg">Setting up game room...</p>
            <p className="text-purple-200 text-sm mt-2">Room: {gameId}</p>
            {error && (
              <div className="mt-4">
                <p className="text-red-300 text-sm mb-2">{error}</p>
                <Button onClick={() => window.location.reload()} size="sm" className="bg-blue-600 hover:bg-blue-700">
                  <RefreshCw className="h-4 w-4 mr-2" />
                  Refresh Page
                </Button>
              </div>
            )}
          </CardContent>
        </Card>
      </div>
    )
  }

  const currentPlayer = gameState.players.find((p) => p.id === currentPlayerId)
  const allPlayersReady = gameState.players.length > 0 && gameState.players.every((p) => p.hasRevealed)

  return (
    <div className="min-h-screen bg-gradient-to-br from-purple-900 via-blue-900 to-indigo-900 p-4">
      <div className="max-w-6xl mx-auto">
        {/* Header */}
        <div className="text-center mb-8">
          <div className="flex items-center justify-center gap-4 mb-4">
            <h1 className="text-4xl md:text-6xl font-bold text-white">🃏 Kostas Late Poker 🕐</h1>
            <div className="flex items-center gap-2">
              {isConnected ? <Wifi className="h-6 w-6 text-green-400" /> : <WifiOff className="h-6 w-6 text-red-400" />}
              <Badge variant={isConnected ? "default" : "destructive"} className="text-xs">
                {isConnected ? "LIVE" : "OFFLINE"}
              </Badge>
            </div>
          </div>

          {/* Room Info and Actions */}
          <div className="flex flex-col sm:flex-row items-center justify-center gap-4 mb-4">
            <div className="flex items-center gap-2">
              <Badge variant="outline" className="text-lg px-4 py-2 bg-white/10 border-white/30 text-white">
                Room: {gameId}
              </Badge>
              <Button onClick={shareGame} size="sm" className="bg-blue-600 hover:bg-blue-700">
                <Share2 className="h-4 w-4 mr-1" />
                Share
              </Button>
              <Button
                onClick={goHome}
                size="sm"
                variant="outline"
                className="border-white/30 text-white hover:bg-white/10"
              >
                <Home className="h-4 w-4 mr-1" />
                Home
              </Button>
            </div>
          </div>

          <p className="text-lg text-purple-300">Place your bets on how late Kostas will be!</p>

          {/* Share Success Message */}
          {showShareSuccess && (
            <Alert className="mt-4 max-w-md mx-auto bg-green-900/50 border-green-500">
              <Copy className="h-4 w-4" />
              <AlertDescription className="text-green-200">
                Game link copied to clipboard! Share it with your team.
              </AlertDescription>
            </Alert>
          )}
        </div>

        {/* Connection Status */}
        {error && (
          <Alert className="mb-6 bg-red-900/50 border-red-500">
            <AlertDescription className="flex items-center justify-between text-red-200">
              <span>{error}</span>
              <Button
                onClick={reconnect}
                size="sm"
                variant="outline"
                className="border-red-400 text-red-200 hover:bg-red-900/30"
              >
                <RefreshCw className="h-4 w-4 mr-1" />
                Retry
              </Button>
            </AlertDescription>
          </Alert>
        )}

        {/* Game Stats */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-8">
          <Card className="bg-white/10 border-white/20 text-white">
            <CardContent className="flex items-center gap-3 p-4">
              <Users className="h-8 w-8 text-blue-400" />
              <div>
                <p className="text-2xl font-bold">{gameState.players.length}</p>
                <p className="text-sm text-gray-300">Players Online</p>
              </div>
            </CardContent>
          </Card>

          <Card className="bg-white/10 border-white/20 text-white">
            <CardContent className="flex items-center gap-3 p-4">
              <Clock className="h-8 w-8 text-green-400" />
              <div>
                <p className="text-2xl font-bold">
                  {gameState.phase === "finished" ? `${gameState.actualTime}min` : "?"}
                </p>
                <p className="text-sm text-gray-300">Actual Time</p>
              </div>
            </CardContent>
          </Card>

          <Card className="bg-white/10 border-white/20 text-white">
            <CardContent className="flex items-center gap-3 p-4">
              <Trophy className="h-8 w-8 text-yellow-400" />
              <div>
                <p className="text-2xl font-bold">{gameState.winner || "?"}</p>
                <p className="text-sm text-gray-300">Winner</p>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Joining Phase */}
        {gameState.phase === "joining" && (
          <Card className="mb-8 bg-white/10 border-white/20">
            <CardHeader>
              <CardTitle className="text-white">Join the Game</CardTitle>
              <CardDescription className="text-gray-300">
                Add your name to bet on Kostas's lateness
                {gameState.players.length === 0 && (
                  <span className="block mt-2 text-blue-300">
                    💡 Share this link with your team:{" "}
                    <code className="bg-white/20 px-2 py-1 rounded text-xs">/game/{gameId}</code>
                  </span>
                )}
              </CardDescription>
            </CardHeader>
            <CardContent>
              {!currentPlayer && (
                <div className="flex gap-2 mb-4">
                  <Input
                    placeholder="Enter your name..."
                    value={newPlayerName}
                    onChange={(e) => setNewPlayerName(e.target.value)}
                    onKeyPress={(e) => e.key === "Enter" && addPlayer()}
                    className="bg-white/20 border-white/30 text-white placeholder:text-gray-400"
                  />
                  <Button onClick={addPlayer} className="bg-blue-600 hover:bg-blue-700">
                    Join Game
                  </Button>
                </div>
              )}

              {currentPlayer && (
                <Alert className="mb-4 bg-green-900/50 border-green-500">
                  <AlertDescription className="text-green-200">
                    You're in the game as <strong>{currentPlayer.name}</strong>!
                  </AlertDescription>
                </Alert>
              )}

              {gameState.players.length > 0 && (
                <div className="space-y-2 mb-4">
                  <h3 className="text-white font-semibold">Players in game:</h3>
                  <div className="flex flex-wrap gap-2">
                    {gameState.players.map((player) => (
                      <Badge
                        key={player.id}
                        variant={player.id === currentPlayerId ? "default" : "secondary"}
                        className={`${
                          player.id === currentPlayerId ? "bg-blue-600 text-white" : "bg-white/20 text-white"
                        }`}
                      >
                        {player.name} {player.score > 0 && `(${player.score} wins)`}
                        {player.id === currentPlayerId && " (You)"}
                      </Badge>
                    ))}
                  </div>
                </div>
              )}

              {gameState.players.length > 0 && (
                <div className="flex gap-2">
                  <Button onClick={startBetting} className="bg-green-600 hover:bg-green-700">
                    Start Betting Round
                  </Button>
                  <Button
                    onClick={shareGame}
                    variant="outline"
                    className="border-white/30 text-white hover:bg-white/10"
                  >
                    <Share2 className="h-4 w-4 mr-2" />
                    Invite More Players
                  </Button>
                </div>
              )}
            </CardContent>
          </Card>
        )}

        {/* Betting Phase */}
        {gameState.phase === "betting" && (
          <div className="space-y-6">
            <Card className="bg-white/10 border-white/20">
              <CardHeader>
                <CardTitle className="text-white">Place Your Bets!</CardTitle>
                <CardDescription className="text-gray-300">How many minutes late will Kostas be?</CardDescription>
              </CardHeader>
            </Card>

            {/* Current Player Status */}
            {currentPlayer && (
              <Alert
                className={`${currentPlayer.hasRevealed ? "bg-green-900/50 border-green-500" : "bg-blue-900/50 border-blue-500"}`}
              >
                <AlertDescription className={currentPlayer.hasRevealed ? "text-green-200" : "text-blue-200"}>
                  {currentPlayer.hasRevealed
                    ? `You bet ${currentPlayer.bet} minutes! Waiting for other players...`
                    : "Choose your bet below:"}
                </AlertDescription>
              </Alert>
            )}

            {/* Betting Options */}
            {currentPlayer && !currentPlayer.hasRevealed && (
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4 mb-6">
                {TIME_OPTIONS.map((option) => (
                  <Card
                    key={option.value}
                    className="bg-white/10 border-white/20 hover:bg-white/20 cursor-pointer transition-all"
                    onClick={() => placeBet(option.value)}
                  >
                    <CardContent className="p-4 text-center">
                      <div className="text-2xl font-bold text-white mb-2">{option.range}</div>
                      <div className="text-purple-200 text-sm">{option.label}</div>
                    </CardContent>
                  </Card>
                ))}
              </div>
            )}

            {/* Players Status */}
            <div className="grid grid-cols-2 md:grid-cols-4 gap-3 mb-6">
              {gameState.players.map((player) => (
                <Card key={player.id} className="bg-white/10 border-white/20">
                  <CardContent className="p-3 text-center">
                    <Avatar className="w-8 h-8 mx-auto mb-2">
                      <AvatarFallback className="text-xs">{player.name.slice(0, 2).toUpperCase()}</AvatarFallback>
                    </Avatar>
                    <p className="text-white text-sm font-medium">{player.name}</p>
                    <Badge className={`mt-1 text-xs ${player.hasRevealed ? "bg-green-600" : "bg-gray-600"}`}>
                      {player.hasRevealed ? "✓ Ready" : "Thinking..."}
                    </Badge>
                  </CardContent>
                </Card>
              ))}
            </div>

            {/* Progress */}
            <div className="text-center">
              <p className="text-white mb-4">
                {gameState.players.filter((p) => p.hasRevealed).length} / {gameState.players.length} players have bet
              </p>

              {allPlayersReady && (
                <Button onClick={revealBets} className="bg-yellow-600 hover:bg-yellow-700 text-black font-bold">
                  <Eye className="w-4 h-4 mr-2" />
                  Reveal All Bets
                </Button>
              )}
            </div>
          </div>
        )}

        {/* Revealed Phase */}
        {gameState.phase === "revealed" && (
          <div className="space-y-6">
            <Card className="bg-white/10 border-white/20">
              <CardHeader>
                <CardTitle className="text-white">All Bets Revealed!</CardTitle>
                <CardDescription className="text-gray-300">Now we wait for Kostas... ⏰</CardDescription>
              </CardHeader>
            </Card>

            {/* Show all bets */}
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4 mb-6">
              {gameState.players.map((player) => (
                <Card key={player.id} className="bg-white/10 border-white/20">
                  <CardContent className="p-4 text-center">
                    <Avatar className="w-12 h-12 mx-auto mb-3">
                      <AvatarFallback>{player.name.slice(0, 2).toUpperCase()}</AvatarFallback>
                    </Avatar>
                    <h3 className="text-white font-semibold mb-2">{player.name}</h3>
                    <div className="text-2xl font-bold text-yellow-400 mb-1">
                      {player.bet !== null ? `${player.bet} min` : "No bet"}
                    </div>
                    <div className="text-sm text-gray-300">
                      {player.bet !== null ? TIME_OPTIONS.find((opt) => opt.value === player.bet)?.label : ""}
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>

            {/* Finish game options */}
            <Card className="bg-white/10 border-white/20">
              <CardHeader>
                <CardTitle className="text-white">Kostas Has Arrived!</CardTitle>
                <CardDescription className="text-gray-300">How many minutes late was he?</CardDescription>
              </CardHeader>
              <CardContent>
                <div className="grid grid-cols-2 md:grid-cols-4 gap-3">
                  {TIME_OPTIONS.map((option) => (
                    <Button
                      key={option.value}
                      onClick={() => finishGame(option.value)}
                      className="bg-green-600 hover:bg-green-700 text-white"
                    >
                      {option.range}
                    </Button>
                  ))}
                </div>
              </CardContent>
            </Card>
          </div>
        )}

        {/* Finished Phase */}
        {gameState.phase === "finished" && (
          <div className="space-y-6">
            <Card className="bg-gradient-to-r from-yellow-400 to-orange-500 border-0">
              <CardContent className="p-8 text-center">
                <Trophy className="w-16 h-16 mx-auto mb-4 text-yellow-800" />
                <h2 className="text-3xl font-bold text-yellow-900 mb-2">🎉 {gameState.winner} Wins! 🎉</h2>
                <p className="text-yellow-800 text-lg">Kostas was {gameState.actualTime} minutes late</p>
              </CardContent>
            </Card>

            {/* Final Results */}
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4 mb-6">
              {gameState.players
                .sort((a, b) => {
                  if (a.name === gameState.winner) return -1
                  if (b.name === gameState.winner) return 1
                  if (a.bet === null) return 1
                  if (b.bet === null) return -1
                  return Math.abs(a.bet - (gameState.actualTime || 0)) - Math.abs(b.bet - (gameState.actualTime || 0))
                })
                .map((player, index) => (
                  <Card
                    key={player.id}
                    className={`${
                      player.name === gameState.winner
                        ? "bg-gradient-to-br from-yellow-400 to-orange-500 border-yellow-300"
                        : "bg-white/10 border-white/20"
                    }`}
                  >
                    <CardContent className="p-4 text-center">
                      <div className="flex items-center justify-center mb-2">
                        {index === 0 && <Trophy className="w-6 h-6 text-yellow-600 mr-2" />}
                        <Avatar className="w-12 h-12">
                          <AvatarFallback className={player.name === gameState.winner ? "bg-yellow-600" : ""}>
                            {player.name.slice(0, 2).toUpperCase()}
                          </AvatarFallback>
                        </Avatar>
                      </div>
                      <h3
                        className={`font-semibold mb-2 ${player.name === gameState.winner ? "text-yellow-900" : "text-white"}`}
                      >
                        {player.name}
                        {player.id === currentPlayerId && " (You)"}
                      </h3>
                      <div
                        className={`text-xl font-bold mb-1 ${player.name === gameState.winner ? "text-yellow-900" : "text-yellow-400"}`}
                      >
                        Bet: {player.bet !== null ? `${player.bet} min` : "No bet"}
                      </div>
                      <div
                        className={`text-sm ${player.name === gameState.winner ? "text-yellow-800" : "text-gray-300"}`}
                      >
                        {player.bet !== null ? `Off by ${Math.abs(player.bet - (gameState.actualTime || 0))} min` : ""}
                      </div>
                      {player.score > 0 && <Badge className="mt-2 bg-purple-600">{player.score} total wins</Badge>}
                    </CardContent>
                  </Card>
                ))}
            </div>

            <div className="text-center space-y-2">
              <Button onClick={resetGame} className="bg-blue-600 hover:bg-blue-700 text-white mr-2">
                <RotateCcw className="w-4 h-4 mr-2" />
                Play Another Round
              </Button>
              <Button onClick={shareGame} variant="outline" className="border-white/30 text-white hover:bg-white/10">
                <Share2 className="h-4 w-4 mr-2" />
                Share Game
              </Button>
            </div>
          </div>
        )}
      </div>
    </div>
  )
}
