"use client"

import { useState, useEffect, useCallback, useRef } from "react"

interface Player {
  id: string
  name: string
  bet: number | null
  hasRevealed: boolean
  score: number
  joinedAt: number
}

interface GameState {
  id: string
  phase: "joining" | "betting" | "revealed" | "finished"
  players: Player[]
  actualTime: number | null
  winner: string | null
  createdAt: number
  lastActivity: number
}

export function useGame(gameId: string) {
  const [gameState, setGameState] = useState<GameState | null>(null)
  const [isConnected, setIsConnected] = useState(false)
  const [error, setError] = useState<string | null>(null)
  const [isInitialized, setIsInitialized] = useState(false)
  const eventSourceRef = useRef<EventSource | null>(null)
  const reconnectTimeoutRef = useRef<NodeJS.Timeout | null>(null)
  const reconnectAttempts = useRef(0)
  const maxReconnectAttempts = 5

  const cleanup = useCallback(() => {
    if (eventSourceRef.current) {
      eventSourceRef.current.close()
      eventSourceRef.current = null
    }
    if (reconnectTimeoutRef.current) {
      clearTimeout(reconnectTimeoutRef.current)
      reconnectTimeoutRef.current = null
    }
  }, [])

  const setupSSE = useCallback(() => {
    if (eventSourceRef.current) {
      return // Already connected
    }

    try {
      const eventSource = new EventSource(`/api/games/${gameId}/events`)
      eventSourceRef.current = eventSource

      eventSource.onopen = () => {
        setIsConnected(true)
        setError(null)
        reconnectAttempts.current = 0
      }

      eventSource.onmessage = (event) => {
        try {
          const data = JSON.parse(event.data)
          if (data.type !== "heartbeat") {
            setGameState(data)
            if (!isInitialized) {
              setIsInitialized(true)
            }
          }
        } catch (err) {
          console.error("Error parsing SSE data:", err)
        }
      }

      eventSource.onerror = () => {
        setIsConnected(false)
        eventSource.close()
        eventSourceRef.current = null

        // Retry connection with exponential backoff
        if (reconnectAttempts.current < maxReconnectAttempts) {
          const delay = Math.min(1000 * Math.pow(2, reconnectAttempts.current), 10000)
          setError(`Connection lost. Reconnecting in ${Math.ceil(delay / 1000)}s...`)

          reconnectTimeoutRef.current = setTimeout(() => {
            reconnectAttempts.current++
            setupSSE()
          }, delay)
        } else {
          setError("Connection failed. Please refresh the page.")
        }
      }
    } catch (err) {
      setError("Failed to establish connection")
      console.error("SSE setup error:", err)
    }
  }, [gameId, isInitialized])

  // Initialize game and setup SSE connection
  useEffect(() => {
    let mounted = true

    const initGame = async () => {
      try {
        // First, ensure the game exists by fetching/creating it
        const response = await fetch(`/api/games/${gameId}`, {
          method: "GET",
          headers: {
            "Cache-Control": "no-cache",
          },
        })

        if (!response.ok) {
          throw new Error(`Failed to initialize game: ${response.status}`)
        }

        const game = await response.json()

        if (mounted) {
          setGameState(game)
          setIsInitialized(true)

          // Small delay to ensure game is fully created before SSE connection
          setTimeout(() => {
            if (mounted) {
              setupSSE()
            }
          }, 100)
        }
      } catch (err) {
        if (mounted) {
          setError("Failed to initialize game")
          console.error("Game initialization error:", err)
        }
      }
    }

    initGame()

    return () => {
      mounted = false
      cleanup()
    }
  }, [gameId, setupSSE, cleanup])

  const sendAction = useCallback(
    async (action: string, payload: any = {}) => {
      try {
        const response = await fetch(`/api/games/${gameId}`, {
          method: "POST",
          headers: {
            "Content-Type": "application/json",
          },
          body: JSON.stringify({ action, payload }),
        })

        if (!response.ok) {
          throw new Error(`Failed to send action: ${response.status}`)
        }

        const updatedGame = await response.json()
        return updatedGame
      } catch (err) {
        setError("Failed to perform action")
        console.error("Action error:", err)
        throw err
      }
    },
    [gameId],
  )

  const reconnect = useCallback(() => {
    cleanup()
    reconnectAttempts.current = 0
    setError(null)
    setupSSE()
  }, [cleanup, setupSSE])

  return {
    gameState,
    isConnected,
    error,
    isInitialized,
    sendAction,
    reconnect,
  }
}
