// Simple in-memory store for game state
interface Player {
  id: string
  name: string
  bet: number | null
  hasRevealed: boolean
  score: number
  joinedAt: number
}

interface GameState {
  id: string
  phase: "joining" | "betting" | "revealed" | "finished"
  players: Player[]
  actualTime: number | null
  winner: string | null
  createdAt: number
  lastActivity: number
}

class GameStore {
  private games = new Map<string, GameState>()
  private subscribers = new Map<string, Set<(game: GameState) => void>>()

  createGame(gameId: string): GameState {
    const game: GameState = {
      id: gameId,
      phase: "joining",
      players: [],
      actualTime: null,
      winner: null,
      createdAt: Date.now(),
      lastActivity: Date.now(),
    }

    this.games.set(gameId, game)
    return game
  }

  getGame(gameId: string): GameState | null {
    return this.games.get(gameId) || null
  }

  updateGame(gameId: string, updater: (game: GameState) => GameState): GameState | null {
    const game = this.games.get(gameId)
    if (!game) return null

    const updatedGame = {
      ...updater(game),
      lastActivity: Date.now(),
    }

    this.games.set(gameId, updatedGame)
    this.notifySubscribers(gameId, updatedGame)
    return updatedGame
  }

  subscribe(gameId: string, callback: (game: GameState) => void): () => void {
    if (!this.subscribers.has(gameId)) {
      this.subscribers.set(gameId, new Set())
    }

    this.subscribers.get(gameId)!.add(callback)

    return () => {
      const subs = this.subscribers.get(gameId)
      if (subs) {
        subs.delete(callback)
        if (subs.size === 0) {
          this.subscribers.delete(gameId)
        }
      }
    }
  }

  private notifySubscribers(gameId: string, game: GameState) {
    const subs = this.subscribers.get(gameId)
    if (subs) {
      subs.forEach((callback) => callback(game))
    }
  }

  // Cleanup old games (older than 2 hours)
  cleanup() {
    const cutoff = Date.now() - 2 * 60 * 60 * 1000
    for (const [gameId, game] of this.games.entries()) {
      if (game.lastActivity < cutoff) {
        this.games.delete(gameId)
        this.subscribers.delete(gameId)
      }
    }
  }
}

export const gameStore = new GameStore()

// Cleanup every 30 minutes
setInterval(() => gameStore.cleanup(), 30 * 60 * 1000)
